/* Copyright (c) 2017-2023 VMware, Inc. All rights reserved. -- VMware Confidential */
import {ProjectMigrator} from "./fileSystem/projectMigrator";
import * as yargs from "yargs";
import * as fs from "fs";
import * as path from "path";

const PROJECT_ARG: string = "project";
const MIGRATION_FILE_ARG: string = "migrationFile";
const MIGRATION_DESTINATION_ARG = "migrationDestination";
const OVERWRITE: string = "overwrite";

const options: { [key: string]: yargs.Options } = {
   project: {
      alias: "ms",
      describe: "The directory of the project which will be migrated.",
      demandOption: true
   },
   migrationFile: {
      alias: "mf",
      describe: "The destination where the migration file will be created.",
      demandOption: true
   },
   migrationDestination: {
      alias: "md",
      describe: "The destination directory where the project will be migrated.",
      demandOption: true
   },
   overwrite: {
      alias: "ow",
      describe: "overwrite existing files.",
      demandOption: false
   }
};

const commandDescription: string = "Migrate a project to a new one in which all calls to the bridge APIs are replaced with calls to the new JS APIs.\n" +
   "The migration can be done immediately or a migration patch file can be generated, edited and applied.\n" +
   "The original project is not edited. ";

yargs
   .usage(commandDescription)
   .command("migrateProject",
      "Copies the project to a new directory and migrates all Bridge API calls " +
      "to the new JS APIs in all .js, .ts and .html files. " +
      "The calls to the old Bridge APIs in the plugin.xml are also migrated.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_DESTINATION_ARG, options.migrationDestination)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         migrateProject(argv.project, argv.migrationDestination, argv.overwrite);
      })
   .command("generateMigrationFile",
      "Generates a migration patch file that can be used to migrate the project. " +
      "The file can be edited before applying it.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_FILE_ARG, options.migrationFile)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         generateMigrationFile(argv.project, argv.migrationFile, argv.overwrite);
      })
   .command("applyMigrationFile",
      "Migrates the project to a new directory with applied migration patch file.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_DESTINATION_ARG, options.migrationDestination)
            .option(MIGRATION_FILE_ARG, options.migrationFile)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         applyMigrationFile(argv.project, argv.migrationFile, argv.migrationDestination, argv.overwrite);
      })
   .example('npm run -s migrate migrateProject -- --project="C:\\hello\\world-html" --migrationDestination="output"\n',
      'Migrates the project world-html into the destination directory output.')
   .example('npm run -s migrate generateMigrationFile -- --project="C:\\hello\\world-html" --migrationFile="diff.txt"\n',
      'Generates a migration diff file diff.txt for the world-html project.')
   .example('npm run -s migrate applyMigrationFile -- --project="C:\\hello\\world-html" --migrationDestination="output" --migrationFile="diff.txt"',
      'Copies the project world-html to the destination directory output and applies the changes from the migration diff file diff.txt')
   .demandCommand().argv;

function generateMigrationFile(projectPath: string, migrationFile: string, overwrite: boolean): void {
   const normalizedProjectPath = path.normalize(projectPath);
   const normalizedMigrationFile = path.normalize(migrationFile);

   if (!directoryExists(normalizedProjectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if(!directoryExists(path.resolve(path.parse(normalizedMigrationFile).dir))) {
      console.log(`The directory '${path.parse(migrationFile).dir}' does not exists.`);
      return;
   }

   if (fileExists(normalizedMigrationFile) && !overwrite) {
      console.log(`The file '${migrationFile}' already exists. Use --overwrite to overwrite.`);
      return;
   }

   if (fileExists(normalizedMigrationFile)) {
      fs.unlinkSync(normalizedMigrationFile);
   }

   const project = new ProjectMigrator(normalizedProjectPath);
   project.createProjectTransformations(normalizedMigrationFile);
}

function applyMigrationFile(projectPath: string, migrationFile: string, projectDestinationPath: string, overwrite: boolean): void {
   const normalizedProjectPath = path.normalize(projectPath);
   const normalizedMigrationFile = path.normalize(migrationFile);
   const normalizedProjectDestinationPath = path.normalize(projectDestinationPath);

   if (!directoryExists(normalizedProjectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if (!fileExists(normalizedMigrationFile)) {
      console.log(`The migration patch file '${migrationFile}' does not exist, generate it with the generateMigrationFile command.`);
      return;
   }

   if(!directoryExists(path.resolve(path.parse(normalizedProjectDestinationPath).dir))) {
      console.log(`The directory '${path.parse(projectDestinationPath).dir}' does not exists.`);
      return;
   }

   if (overwrite) {
      deleteFolderRecursive(normalizedProjectDestinationPath);
   }

   if (directoryExists(normalizedProjectDestinationPath) && !isDirEmpty(normalizedProjectDestinationPath)) {
      console.log(`The directory '${projectDestinationPath}' is not empty. Use --overwrite to overwrite.`);
      return;
   }

   ProjectMigrator.copyProject(normalizedProjectPath, normalizedProjectDestinationPath)
      .then(() => {
         const project = new ProjectMigrator(normalizedProjectDestinationPath);
         try {
            project.applyProjectTransformations(normalizedMigrationFile);
         } catch(e) {
            deleteFolderRecursive(normalizedProjectDestinationPath);
            console.error(e);
         }
      });
}

function migrateProject(projectPath: string, projectDestinationPath: string, overwrite: boolean): void {
   const normalizedProjectPath = path.normalize(projectPath);
   const normalizedProjectDestinationPath = path.normalize(projectDestinationPath);

   if (!directoryExists(normalizedProjectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if(!directoryExists(path.resolve(path.parse(normalizedProjectDestinationPath).dir))) {
      console.log(`The directory '${path.parse(projectDestinationPath).dir}' does not exists.`);
      return;
   }

   if (overwrite) {
      deleteFolderRecursive(normalizedProjectDestinationPath);
   }

   if (directoryExists(normalizedProjectDestinationPath) && !isDirEmpty(normalizedProjectDestinationPath)) {
      console.log(`The directory '${projectDestinationPath}' is not empty. Use --overwrite to overwrite.`);
      return;
   }

   ProjectMigrator.copyProject(normalizedProjectPath, normalizedProjectDestinationPath)
      .then(() => {
         const project = new ProjectMigrator(normalizedProjectDestinationPath);
         try {
            project.migrateProject();
         } catch(e) {
            deleteFolderRecursive(normalizedProjectDestinationPath);
            console.error(e);
         }
      });
}

function directoryExists(path: string): boolean {
   if (!(fs.existsSync(path) && fs.lstatSync(path).isDirectory())) {
      return false;
   }
   return true;
}

function fileExists(path: string): boolean {
   return ((fs.existsSync(path) && !fs.lstatSync(path).isDirectory()));
}

function isDirEmpty(path: string): boolean {
   return fs.readdirSync(path).length == 0;
}

function deleteFolderRecursive(path: string) {
   if (fs.existsSync(path)) {
      fs.readdirSync(path).forEach(function (file, index) {
         const curPath = path + "/" + file;
         if (fs.lstatSync(curPath).isDirectory()) { // recurse
            deleteFolderRecursive(curPath);
         } else { // delete file
            fs.unlinkSync(curPath);
         }
      });
      fs.rmdirSync(path);
   }
}
